/**
* \file: message_buffer.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>

#include "ipc/message_buffer.h"
#include "utils/logger.h"

#ifdef UNIT_TESTING_ENABLE_SEND_RECV_WRAPPER

//send and recv wrapper only enabled for unit testing
extern ssize_t send_wrapper(int socket, void *buffer, size_t buffer_size, int flags);

extern ssize_t recv_wrapper(int socket, void *buffer, size_t buffer_size, int flags);

#endif


//------------------------------------- private type definitions -----------------------------------------------------

typedef struct message_buffer_t
{
	//the size of the buffer
	size_t buffer_size;

	//the buffer itself
	void *buffer;

	//the state of the buffer
	message_buffer_state_t state;

	// the number of bytes already processed (sent or received)
	// In all states despite MSG_BUFFER_SENDING and MSG_BUFFER_RECEIVING this attribute is always set to 0
	size_t bytes_processed;

	// used to iterate through the strings of a message
	void *string_element_iterator;

	//ptr to the next and previous buffers. These ptrs are used to realize linked lists (in sending queues for instance)
	message_buffer_t *next_buffer;
	message_buffer_t *previous_buffer;
} message_buffer_t;


/**
 * message memory mapping:
 * | message_t | data | string_t[] |
 * message_t::msg_size -> overall message length (bytes)
 * message_t::data_size -> size of data part (bytes)
 *
 * for each string
 * 		string_t::string_buffer_len -> the length of the string including this element
 * 		string_t::string
 * end
 */

typedef struct string_t
{
	size_t string_buffer_len;

	char string[];
} string_t;

typedef struct message_t
{
	size_t msg_size;

	message_type_t type;

	size_t data_size;

	char data[];
}  message_t;
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ---------------------------------------------------
static error_code_t message_buffer_check_size_and_increase_if_needed(message_buffer_t *msg_buffer, size_t new_size);

static error_code_t message_buffer_do_send(int a_socket, message_buffer_t *msg_buffer);

static error_code_t message_buffer_do_receive(message_buffer_t *msg_buffer,size_t expected_size,int a_socket);
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - buffer mgnt --------------------------------------------------
message_buffer_t *message_buffer_create_new(size_t initial_size)
{
	message_buffer_t *msg_buffer;

	//at least the message header must fit into each buffer
	if (initial_size<sizeof(message_t))
		initial_size=sizeof(message_t);

	msg_buffer=malloc(sizeof(message_buffer_t));
	if (msg_buffer != NULL)
	{
		msg_buffer->buffer=malloc(initial_size);
		msg_buffer->buffer_size=initial_size;
		msg_buffer->next_buffer=NULL;
		msg_buffer->previous_buffer=NULL;
		msg_buffer->state=MSG_BUFFER_EMPTY;
		msg_buffer->bytes_processed=0;
		msg_buffer->string_element_iterator=NULL;
	}
	else
		return NULL;

	if (msg_buffer->buffer==NULL)
	{
		message_buffer_destroy(msg_buffer);
		msg_buffer=NULL;
	}

	return msg_buffer;
}

void message_buffer_destroy(message_buffer_t *msg_buffer)
{
	if (msg_buffer != NULL)
	{
		if (msg_buffer->buffer!=NULL)
			free(msg_buffer->buffer);
		free(msg_buffer);
	}
}

void message_buffer_mark_empty(message_buffer_t *msg_buffer)
{
	if (msg_buffer != NULL)
	{
		msg_buffer->state=MSG_BUFFER_EMPTY;
		msg_buffer->string_element_iterator=NULL;
	}
}

void message_buffer_reuse_message_for_sending(message_buffer_t *msg_buffer)
{
	if (msg_buffer->state==MSG_BUFFER_MSG_SENT)
	{
		msg_buffer->state=MSG_BUFFER_MSG_PREPARED_FOR_SENDING;
		msg_buffer->bytes_processed=0;
	}
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - msg queue ----------------------------------------------------
void message_buffer_init_queue(message_queue_t *queue)
{
	queue->first_buffer=NULL;
	queue->last_buffer=NULL;
	queue->no_elements=0;
}

void message_buffer_insert_at_begining(message_queue_t *queue, message_buffer_t *message_buffer)
{
	if (message_buffer->next_buffer!=NULL || message_buffer->previous_buffer!=NULL)
	{
		logger_log_error("MESSAGE_BUFFER - Implementation issue. A message buffer is tried to be linked into two queues."
				" This is not allowed since this will destroy one of both lists.");
		return;
	}

	//update the forward chain
	message_buffer->next_buffer=queue->first_buffer;
	queue->first_buffer=message_buffer;

	//update the backward chain
	if (message_buffer->next_buffer!=NULL)
		message_buffer->next_buffer->previous_buffer=message_buffer;
	else
		//we are the first one in the queue, set last_buffer accordingly
		queue->last_buffer=message_buffer;

	queue->no_elements++;
}

message_buffer_t  *message_buffer_deque_from_begining(message_queue_t *queue)
{
	message_buffer_t *buf;
	if (queue->first_buffer==NULL)
		return NULL;

	//store the first buffer
	buf=queue->first_buffer;

	//make the second one the first one
	queue->first_buffer=buf->next_buffer;

	if (queue->first_buffer!=NULL)
		//if we have a new first buffer ...
		queue->first_buffer->previous_buffer=NULL;
	else
		//otherwise we are empty
		queue->last_buffer=NULL;

	queue->no_elements--;

	buf->next_buffer=NULL;

	return buf;
}

void message_buffer_remove_from_queue(message_queue_t *queue, message_buffer_t *message_buffer)
{
	//update queue start and end ptr
	if (message_buffer==queue->first_buffer)
		queue->first_buffer=message_buffer->next_buffer;
	if (message_buffer==queue->last_buffer)
		queue->last_buffer=message_buffer->previous_buffer;

	//unchain forward links
	if (message_buffer->previous_buffer != NULL)
		message_buffer->previous_buffer->next_buffer=message_buffer->next_buffer;

	//unchain backward links
	if (message_buffer->next_buffer != NULL)
		message_buffer->next_buffer->previous_buffer=message_buffer->previous_buffer;

	//mark the buffer as unqueued
	message_buffer->next_buffer=NULL;
	message_buffer->previous_buffer=NULL;

	queue->no_elements--;
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - sending ------------------------------------------------------
error_code_t message_buffer_prepare_message(message_buffer_t *msg_buffer, message_type_t message_type,
		size_t data_size, void **data_buffer)
{
	error_code_t result;
	size_t msg_size;

	if (msg_buffer->state!=MSG_BUFFER_EMPTY)
		return RESULT_INVALID;

	msg_size=sizeof(message_t)+data_size;
	result=message_buffer_check_size_and_increase_if_needed(msg_buffer, msg_size);

	if (result==RESULT_OK)
	{
		message_t *msg=(message_t *)msg_buffer->buffer;
		msg->type=message_type;
		msg->data_size=data_size;
		msg->msg_size=msg_size;
		*data_buffer=msg->data;

		//buffer changes its state from MSG_BUFFER_EMPTY -> MSG_BUFFER_MSG_PREPARED_FOR_SENDING
		msg_buffer->state=MSG_BUFFER_MSG_PREPARED_FOR_SENDING;
		msg_buffer->string_element_iterator=NULL;
	}

	return result;
}

error_code_t message_buffer_add_string(message_buffer_t *msg_buffer, const char *string)
{
	size_t string_buf_len;
	size_t msg_size;
	message_t *msg;
	error_code_t result;

	if (msg_buffer->state!=MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
		return RESULT_INVALID;

	if (string==NULL)
		string="";

	//since we are in state MSG_BUFFER_MSG_PREPARED_FOR_SENDING, someone prepared a message for us and placed it
	//into msg_buffer->buffer
	msg=(message_t *)msg_buffer->buffer;

	//buffer needed equals the size information + the length of the string + 1 (for '\0')
	string_buf_len=sizeof(string_t)+strlen(string)+1;
	//new message size is the old one plus the size needed for the new string
	msg_size=msg->msg_size+string_buf_len;

	result=message_buffer_check_size_and_increase_if_needed(msg_buffer, msg_size);
	//msg_buffer->buffer might have been reallocated
	msg=(message_t *)msg_buffer->buffer;

	if (result==RESULT_OK)
	{
		//the next string is to be placed at the end of the message
		// sorry, lin(t) forces me to do it a bit complicated
		string_t *next_string_ptr=(void *)(((char *)msg)+msg->msg_size);
		next_string_ptr->string_buffer_len=string_buf_len;
		strcpy(next_string_ptr->string,string);
		msg->msg_size=msg_size;
	}

	return result;
}

error_code_t message_buffer_send(message_buffer_t *msg_buffer, int a_socket)
{
	error_code_t result;

	if (msg_buffer->state != MSG_BUFFER_MSG_PREPARED_FOR_SENDING &&
		msg_buffer->state != MSG_BUFFER_SENDING)
		return RESULT_INVALID;

	//we got a new message, reset the "bytes processed" counter and set the state to sending
	if (msg_buffer->state == MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
	{
		msg_buffer->bytes_processed=0;
		msg_buffer->state=MSG_BUFFER_SENDING;
	}

	result=message_buffer_do_send(a_socket, msg_buffer);
	if (result==RESULT_OK)
		msg_buffer->state=MSG_BUFFER_MSG_SENT;

	return result;
}

/**
 * this function clones the message currently stored in src_buffer. src_buffer needs to be in state MSG_BUFFER_SENDING
 * or in state MSG_BUFFER_PREPARED_FOR_SENDING. The dst_buffer needs to be in state MSG_BUFFER_EMPTY. After cloning
 * successfully the message, the dst_buffer will be in state MSG_PREPARED_FOR_SENDING.
 *
 * This function is used to send broadcast messages. In case the message couldn't be sent completely to one of the
 * receivers, a copy of this one can be taken and used to send the message to other ones while the first buffer is
 * queued into the sender queue.
 */
error_code_t message_buffer_clone_msg_in_buffer(message_buffer_t *dst_buffer, message_buffer_t *src_buffer)
{
	message_t *msg;
	error_code_t result;

	if ((src_buffer->state != MSG_BUFFER_MSG_PREPARED_FOR_SENDING && src_buffer->state != MSG_BUFFER_SENDING) ||
			dst_buffer->state != MSG_BUFFER_EMPTY)
		return RESULT_INVALID;

	//state of src buffer ensures we have a complete message in it
	msg=(message_t *)src_buffer->buffer;
	result=message_buffer_check_size_and_increase_if_needed(dst_buffer,msg->msg_size);
	if (result==RESULT_OK)
	{
		memcpy(dst_buffer->buffer, msg,msg->msg_size);
		dst_buffer->state=MSG_BUFFER_MSG_PREPARED_FOR_SENDING;
		dst_buffer->bytes_processed=0;
		dst_buffer->string_element_iterator=NULL;
	}
	return result;
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member - receiving ----------------------------------------------------
error_code_t message_buffer_receive(message_buffer_t *msg_buffer, int a_socket)
{
	error_code_t result=RESULT_OK;
	message_t *msg;

	if (msg_buffer->state != MSG_BUFFER_EMPTY &&
		msg_buffer->state != MSG_BUFFER_RECEIVING)
		return RESULT_INVALID;

	if (msg_buffer->state == MSG_BUFFER_EMPTY)
	{
		msg_buffer->bytes_processed=0;
		msg_buffer->state = MSG_BUFFER_RECEIVING;
	}

	msg_buffer->string_element_iterator=NULL;

	//two step approach:
	// - read out the message header (message_t), we will find the actual message size in it
	// - read out the complete message, size is known from the msg_size field

	// we don't need to check for the buffer size. It is ensured during the buffer creation that at least message_t
	// fits into it.

	if (msg_buffer->bytes_processed < sizeof(message_t))
		//we are working on the message header
		result=message_buffer_do_receive(msg_buffer,sizeof(message_t),a_socket);

	if (result!=RESULT_OK)
		//we either got a first part of the message header only or an connection error. The actual reason is encoded in
		//"result". In both cases we can stop here.
		return result;

	//We are getting here when we read out the header completely. Several message_buffer_received calls might have been
	//involved. We can now start reading the rest.
	msg=(message_t *)msg_buffer->buffer;

	//check if our message fits into the buffer. If not, exceed the buffer
	result=message_buffer_check_size_and_increase_if_needed(msg_buffer, msg->msg_size);

	//msg_buffer->buffer might have been reallocated
	msg=(message_t *)msg_buffer->buffer;

	if (result!=RESULT_OK)
		return RESULT_NORESOURCE;

	result=message_buffer_do_receive(msg_buffer, msg->msg_size,a_socket);

	if (result==RESULT_OK)
		//we got the complete message
		msg_buffer->state = MSG_BUFFER_MSG_RECEIVED;

	return result;
}

message_type_t message_buffer_get_msg_type(message_buffer_t *msg_buffer)
{
	if (msg_buffer->state!=MSG_BUFFER_MSG_RECEIVED &&
		msg_buffer->state!=MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
		return __NO_MSG;

	return ((message_t *)(msg_buffer->buffer))->type;
}

void message_buffer_get_msg_data(message_buffer_t *msg_buffer, void **buf_ptr, size_t *buffer_size_ptr)
{
	message_t *msg;

	if (msg_buffer->state!=MSG_BUFFER_MSG_RECEIVED &&
		msg_buffer->state!=MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
	{
		*buf_ptr=NULL;
		*buffer_size_ptr=0;
		return;
	}

	msg=(message_t *)(msg_buffer->buffer);

	*buf_ptr=msg->data;
	*buffer_size_ptr=msg->data_size;
}

const char *message_buffer_get_first_msg_string(message_buffer_t *msg_buffer)
{
	message_t *msg;

	if (msg_buffer->state!=MSG_BUFFER_MSG_RECEIVED &&
		msg_buffer->state!=MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
		return NULL;

	msg=(message_t *)(msg_buffer->buffer);

	//the string list starts after the fixed part of the message and the fixed data structure
	msg_buffer->string_element_iterator=((char *)(msg_buffer->buffer))+sizeof(message_t)+msg->data_size;
	if ((size_t)((char *)(msg_buffer->string_element_iterator)-(char *)msg_buffer->buffer) >= msg->msg_size)
	{
		//element pointer ran out the memory of the message, so we obviously reached the last string before
		msg_buffer->string_element_iterator=NULL;
		return NULL;
	}
	else
		return ((string_t *)msg_buffer->string_element_iterator)->string;
}

const char *message_buffer_get_next_msg_string(message_buffer_t *msg_buffer)
{
	message_t *msg;
	void *next_string_element_ptr;

	if (msg_buffer->state!=MSG_BUFFER_MSG_RECEIVED &&
		msg_buffer->state!=MSG_BUFFER_MSG_PREPARED_FOR_SENDING)
		return NULL;

	// iterator not set to first string -> return NULL
	if (msg_buffer->string_element_iterator==NULL)
		return NULL;

	next_string_element_ptr=(char *)(msg_buffer->string_element_iterator)+
			((string_t *)msg_buffer->string_element_iterator)->string_buffer_len;

	msg=(message_t *)msg_buffer->buffer;
	if ((size_t)((char *)next_string_element_ptr-(char *)(msg_buffer->buffer)) >= msg->msg_size)
	{
		//element pointer ran out the memory of the message, so we obviously reached the last string before
		msg_buffer->string_element_iterator=NULL;
		return NULL;
	}
	else
		msg_buffer->string_element_iterator=next_string_element_ptr;


	return ((string_t *)next_string_element_ptr)->string;
}
//--------------------------------------------------------------------------------------------------------------------

//--------------------------------- private member functions ---------------------------------------------------------
static error_code_t message_buffer_check_size_and_increase_if_needed(message_buffer_t *msg_buffer, size_t new_size)
{
	void *new_buf;
	if (new_size < msg_buffer->buffer_size)
		return RESULT_OK;

	new_buf=realloc(msg_buffer->buffer,new_size);
	if (new_buf==NULL)
		return RESULT_NORESOURCE;
	msg_buffer->buffer=new_buf;
	msg_buffer->buffer_size=new_size;
	return RESULT_OK;
}

static error_code_t message_buffer_do_send(int a_socket, message_buffer_t *msg_buffer)
{
	message_t *msg;
	ssize_t bytes_sent;
	size_t bytes_to_send;
	char *sending_data_ptr;

	//since we are in state MSG_BUFFER_MSG_SENDING (ensured by the caller), someone prepared a message
	//for us and placed it into msg_buffer->buffer
	msg=(message_t *)msg_buffer->buffer;

	//the number of bytes to send is the size of the message minus the bytes already processed in a previous call
	bytes_to_send=msg->msg_size-msg_buffer->bytes_processed;
	//the pointer where to take data to send from is the ptr of the message + the bytes already sent before
	sending_data_ptr=((char *)msg)+msg_buffer->bytes_processed;

#ifdef UNIT_TESTING_ENABLE_SEND_RECV_WRAPPER
	bytes_sent=send_wrapper(a_socket, sending_data_ptr, bytes_to_send, 0);
#else
	bytes_sent=send(a_socket, sending_data_ptr, bytes_to_send, 0);
#endif

	// nothing could be sent
	if (bytes_sent<0)
	{
		// the connection is active but the socket buffer is unable to take additional data (errno==EAGAIN ||
		// errno==EWOULDBLOCK), we return this result respectively.
		if (errno==EAGAIN || errno==EWOULDBLOCK )
			return RESULT_MSG_PARTLY_PROCESSED;
		else
			//otherwise, we take it as a connection error / lost connection / ...
			return RESULT_SOCKET_ERR;
	}

	msg_buffer->bytes_processed+=(size_t)bytes_sent;
	if (msg_buffer->bytes_processed>=msg->msg_size)
		return RESULT_OK;
	else
		return RESULT_MSG_PARTLY_PROCESSED;
}

static error_code_t message_buffer_do_receive(message_buffer_t *msg_buffer,size_t expected_size,int a_socket)
{
	ssize_t bytes_received;
	size_t bytes_to_receive;
	void *receiving_data_ptr;

	if (msg_buffer->bytes_processed>=expected_size)
		//we already read the expected size. We should never get here
		return RESULT_INVALID;


	//the number of bytes to receive is expected number if bytes minus the bytes already processed in a previous call
	bytes_to_receive=expected_size-msg_buffer->bytes_processed;
	//the pointer where to put the received data is the ptr of the buffer + the bytes already received before
	receiving_data_ptr=((char *)(msg_buffer->buffer)+msg_buffer->bytes_processed);

#ifdef UNIT_TESTING_ENABLE_SEND_RECV_WRAPPER
	bytes_received=recv_wrapper(a_socket,receiving_data_ptr,bytes_to_receive,MSG_DONTWAIT);
#else
	bytes_received=recv(a_socket,receiving_data_ptr,bytes_to_receive,MSG_DONTWAIT);
#endif
	//check if we got called without data in the socket buffer
	if (bytes_received==-1)
	{
		if (errno==EAGAIN || errno==EWOULDBLOCK)
			//we didn't get some data that's fine
			return RESULT_MSG_PARTLY_PROCESSED;
		else
			//we ran into another error, take it as a connection error
			return RESULT_SOCKET_ERR;
	}

	// we are getting 0 in case the socket was closed by the other side
	if (bytes_received==0)
		return RESULT_SOCKET_CLOSED;

	//we got some actual data, add it to the number of bytes already read
	msg_buffer->bytes_processed+=(size_t)bytes_received;

	//check the expected size against the received data
	if (msg_buffer->bytes_processed>=expected_size)
		return RESULT_OK;
	else
		return RESULT_MSG_PARTLY_PROCESSED;
}
//--------------------------------------------------------------------------------------------------------------------
